#include "RunDigDagDialog.hpp"
#include "ui_RunDigDagDialog.h"
#include <QtGui>
#include <QStringList>

//===========================================================================================
// Methods

RunDigDagDialog::RunDigDagDialog(QWidget *parent) :
    QDialog(parent),
    _ui(new Ui::RunDigDagDialog) {

  _ui->setupUi(this);

  connectSlots();

}


RunDigDagDialog::~RunDigDagDialog() {
  delete _ui;
}


void RunDigDagDialog::connectSlots() {
  QObject::connect(_ui->customPathButton, SIGNAL(clicked()), this, SLOT(openDigDagPathDialog()));
  QObject::connect(_ui->inputFolderButton, SIGNAL(clicked()), this, SLOT(openInputFolderDialog()));
  QObject::connect(_ui->outputFileButton, SIGNAL(clicked()), this, SLOT(openOutputFolderDialog()));
  QObject::connect(_ui->automaticNameCheckBox, SIGNAL(clicked(bool)), this, SLOT(checkAutomaticOutputName(bool)));
}


void RunDigDagDialog::updateAutomaticOutputFileName() {
  if (_ui->automaticNameCheckBox->isChecked()) {
    QString newName = QString("t%1_g%2")
                      .arg(_ui->minSupportSpinBox->value())
                      .arg(_ui->epThresholdSpinBox->value());
    _ui->outputFileNameLineEdit->setText(newName);
  }
}


//===========================================================================================
// Getters & setters

QString RunDigDagDialog::getDigDagPath() {

  if (_ui->customPathRadioButton->isChecked()) {
    // Return custom path value
    return _ui->customPathLineEdit->text();
  } else {
    // TODO retrieve prefs path value and return it
    return QString("invalid_path");
  }

}


QString RunDigDagDialog::getInputFolderName() {
  return _ui->inputFolderLineEdit->text();
}


bool RunDigDagDialog::isSortEnabled() {
  return _ui->sortFilesCheckBox->isChecked();
}


int RunDigDagDialog::getMinSupport() {
  return _ui->minSupportSpinBox->value();
}


int RunDigDagDialog::getEPThreshold() {
  return _ui->epThresholdSpinBox->value();
}


QString RunDigDagDialog::getOutputPrefix() {
  QString outputPrefix =
      _ui->outputFileLineEdit->text() +
      "/" +
      _ui->outputFileNameLineEdit->text();

  return outputPrefix;
}


int RunDigDagDialog::getFinalAction() {
  int action = -1;
  if (_ui->openRadioButton->isChecked())
    action =  OPEN_FILE;
  else if (_ui->nothingRadioButton)
    action = DO_NOTHING;

  return action;
}


//===========================================================================================
// Private Slots

/**
 * Opens a file chooser dialog to select the DigDag executable
 */
void RunDigDagDialog::openDigDagPathDialog() {

  QFileDialog fileDialog(this, QObject::tr("Select DigDag executable"));
  fileDialog.setFileMode(QFileDialog::ExistingFile);

  QStringList fileNames;
  if (fileDialog.exec()) {

    fileNames = fileDialog.selectedFiles();

    // Update line edit
    _ui->customPathLineEdit->setText(fileNames.at(0));
  }

}


/**
 * Opens a folder chooser dialog to select the input folder containing the input DAG files
 */
void RunDigDagDialog::openInputFolderDialog() {

  QFileDialog fileDialog(this, QObject::tr("Select input folder"));
  fileDialog.setFileMode(QFileDialog::Directory);

  QStringList fileNames;
  if (fileDialog.exec()) {

    fileNames = fileDialog.selectedFiles();

    // Update line edit
    QString inputFolder = fileNames.at(0);
    _ui->inputFolderLineEdit->setText(inputFolder.append("/"));

    // Check folder content, and count number of files
    QDir dir(inputFolder);

    QStringList filters;
    filters << "*.dag";
    dir.setNameFilters(filters);

    // Count number of .dag files
    int _numberOfInputDags = dir.count();
    if (_numberOfInputDags > 0) {
      _ui->inputFolderStatusLabel->setText(tr("%1 DAGs found.").arg(_numberOfInputDags));

      _ui->minSupportHorizontalSlider->setMaximum(_numberOfInputDags);
      _ui->minSupportHorizontalSlider->setEnabled(true);

      _ui->minSupportSpinBox->setMaximum(_numberOfInputDags);
      _ui->minSupportSpinBox->setEnabled(true);

    }
    else {
      _ui->inputFolderStatusLabel->setText(tr("No DAG files in selected folder. Select another one."));
    }

  }

}


/**
 * Opens a folder chooser dialog to select the output folder wich will contain the output file
 */
void RunDigDagDialog::openOutputFolderDialog() {

  QFileDialog fileDialog(this, QObject::tr("Select output folder"));
  fileDialog.setFileMode(QFileDialog::Directory);

  QStringList fileNames;
  if (fileDialog.exec()) {

    fileNames = fileDialog.selectedFiles();

    // Update line edit
    _ui->outputFileLineEdit->setText(fileNames.at(0));
  }

}


void RunDigDagDialog::checkAutomaticOutputName(bool enabled) {

  // Change line edit state
  _ui->outputFileNameLineEdit->setEnabled(!enabled);

  // Change line edit content
  if (enabled) {
    _oldOutputFileName = _ui->outputFileNameLineEdit->text();
    updateAutomaticOutputFileName();
  }
  else {
    _ui->outputFileNameLineEdit->setText(_oldOutputFileName);
  }

}

void RunDigDagDialog::on_minSupportHorizontalSlider_valueChanged(int value) {
  _ui->minSupportSpinBox->setValue(value);
  updateAutomaticOutputFileName();
}



void RunDigDagDialog::on_minSupportSpinBox_valueChanged(int value) {
  _ui->minSupportHorizontalSlider->setValue(value);
  updateAutomaticOutputFileName();
}

void RunDigDagDialog::on_epThresholdSpinBox_valueChanged(int ) {
  updateAutomaticOutputFileName();
}
