#include "DagPattern.hpp"
#include <QObject>

//==============================================================================
// Constructors

/**
 * Default constructor
 */
DagPattern::DagPattern() :
    AbstractDag(),
    _loaded(false),
    _support(0),
    _dagDescriptionPos(0)
{
  // XXX Useless code ?
  // Empty
}


/**
 * Constructor
 */
DagPattern::DagPattern(int index,
                       QString graphName,
                       int numberOfNodes,
                       int numberOfEdges,
                       GVC_t *graphvizContext) :
    AbstractDag(graphName,
                numberOfNodes,
                numberOfEdges,
                graphvizContext),
    _index(index),
    _loaded(false),
    _support(0),
    _dagDescriptionPos(0)
{
  // Empty
}


//==============================================================================
// Getters & setters

/**
 * Returns an int value identifying the DAG Pattern type.
 *
 * @return Integer value identifying the DAG pattern type.
 * @see AbstractDag.hpp
 */
int DagPattern::getDagType() const {
  return AbstractDag::DAG_PATTERN;
}


/**
 * Returns a string describing the DAG type
 *
 * @return A string describing the DAG type
 */
QString DagPattern::getDagTypeString() const {
  return QObject::tr("Pattern");
}


/**
 * Sets this DAG pattern's index.
 *
 * @param dagPatternIndex This DAG pattern's index
 */
void DagPattern::setIndex(int dagPatternIndex) {
  _index = dagPatternIndex;
}


/**
 * Returns this DAG pattern's index.
 *
 * @return This DAG pattern's index
 */
int DagPattern::getIndex() const {
  return _index;
}


/**
 * Sets the boolean loaded state of the DAG pattern :
 * - loaded == true : all pattern information has been read from the DigDag file
 * (general information, DAG structure, TID list).
 * -loaded == false : only general information has been loaded from DigDag file
 * (meaning DAG can not yet be displayed since graph structure is empty).
 *
 * param loaded The boolean loaded state of this DAG pattern
 * @see DigDagData
 */
void DagPattern::setLoaded(bool loaded) {
  _loaded = loaded;
}


/**
 * Returns boolean the loaded state of the DAG pattern :
 * - loaded == true : all pattern information has been read from the DigDag file
 * (general information, DAG structure, TID list).
 * -loaded == false : only general information has been loaded from DigDag file
 * (meaning DAG can not yet be displayed since graph structure is empty).
 *
 * @returns The boolean loaded state of this DAG pattern
 * @see DigDagData
 */
int DagPattern::isLoaded() const {
  return _loaded;
}


/**
 * Sets the support value for this DAG pattern.
 *
 * @param support The support for this DAG pattern
 */
void DagPattern::setSupport(int support) {
  _support = support;
}


/**
 * Returns the support value from this DAG pattern. This value is equivalent to
 * the TID list size.
 *
 * @return Support value from this DAG pattern
 */
int DagPattern::getSupport() const {
  return _support;
}


/**
 * Sets the file position value wich points to the first line describing the graph
 * structure. This is value is necessary in order to remember where to read the
 * DigDag file to complete the DAG pattern information.
 *
 * @param filePositon A file position
 */
void DagPattern::setDagDescriptionPos(qint64 filePosition) {
  _dagDescriptionPos = filePosition;
}


/**
 * Returns the file position that points to the DigDag file line describing the
 * DAG pattern graph structure.
 */
qint64 DagPattern::getDagDescriptionPos() const {
  return _dagDescriptionPos;
}


/**
 * Adds the given input DAG index to the TID list.
 *
 * @param inputDagIndex An input DAG index.
 */
void DagPattern::appendInputDagIndex(int inputDagIndex) {
  _tidList.append(inputDagIndex);
}


/**
 * Returns a pointer to this DAG pattern's TID list.
 *
 * @return The DAG pattern's TID list.
 */
QList<int>* DagPattern::getTidList() {
  return &_tidList;
}
